module talk

//	**************************************************************************************************
//
//	This program creates two interactive processes that communicate via message passing.
//	In a future distributed version this program can be used as a graphical talk application.
//
//	The program has been written in Clean 1.3.1 and uses the Clean Standard Object I/O library 1.0.1
//	
//	**************************************************************************************************

import	StdEnv, StdIO

//	The essential data types. Other data types are given at the end of the program text.
//	The message type of talk processes:
::	Message
	=	NewLine String						// Transmit a line of text
	|	Quit								// Request termination
::	NoState
	=	NoState								// The singleton data type

//	Start creates two talk processes A and B that communicate by means of message passing.
Start :: *World -> *World
Start world
	#	(a,world)	= openRId world
	#	(b,world)	= openRId world
	=	startProcesses (ProcessGroup NoState (ListCS [talk "A" a b,talk "B" b a])) world
where
	talk :: String (RId Message) (RId Message) -> MDIProcess .p
	talk name me you
		=	MDIProcess NoState [initTalk name me you] [ProcessNoWindowMenu]

//	initTalk initialises a talk process.
initTalk :: String (RId Message) (RId Message) (PSt .l .p) -> PSt .l .p
initTalk name me you ps
	#	menu		= Menu ("Talk "+++name)
						(	MenuItem "Quit" [	MenuShortKey 'q'
											,	MenuFunction (noLS (quit you))
											]
						)	[]
	#	(error,ps)	= openMenu undef menu ps
	|	error<>NoError
		=	abort "talk could not open menu."
	#	(wId,  ps)	= accPIO openId ps
	#	(outId,ps)	= accPIO openId ps
	#	(error,ps)	= openReceiver undef (Receiver me (noLS1 (receive wId outId)) []) ps
	|	error<>NoError
		=	abort "talk could not open receiver."
	#	(inId, ps)	= accPIO openId ps
	#	window		= Dialog ("Talk "+++name) 
						(	EditControl	"" (hmm 100.0) 10
									[	ControlId		inId
									,	ControlKeyboard	inputfilter Able (noLS1 (input wId inId you))
									]
						:+:	EditControl	"" (hmm 100.0) 10
									[	ControlId		outId
									,	ControlPos		(BelowPrev,zero)
									,	ControlSelectState Unable
									]
						)
						[	WindowId	wId
						]
	#	(error,ps)	= openDialog undef window ps
	|	error<>NoError
		=	abort "talk could not open window."
	|	otherwise
		=	ps

/*	input handles keyboard input in the input EditControl: 
	for every KeyDown keyboard input that has been accepted by the input EditControl, input sends the 
	current content text of the input EditControl to the other talk process with (NewLine text).
*/
inputfilter :: KeyboardState -> Bool
inputfilter keystate
	=	getKeyboardStateKeyState keystate<>KeyUp

input :: Id Id (RId Message) KeyboardState (PSt .l .p) -> PSt .l .p
input wId inId you _ ps
	#	(Just window,ps)	= accPIO (getWindow wId) ps
		text				= fromJust (snd (hd (getControlTexts [inId] window)))
	=	snd (asyncSend you (NewLine text) ps)
	
/*	The message passing protocol of a talk process.
	On receipt of:
	(1)	NewLine text:set the new text to the output control field of the talk dialog.
	(2) Quit:	     this is always the last message of the other talk process when termination is 
		              requested. The process should terminate itself.
*/
receive :: Id Id Message (PSt .l .p) -> PSt .l .p
receive wId outId (NewLine text) ps=:{io}
	=	{ps & io=setWindow wId  [	setControlTexts    [(outId,text)]
								,	setEditControlCursor outId (size text)
								]	io}
receive _ _ Quit ps
	=	closeProcess ps

/*	The quit command first sends the Quit message to the other talk process and then quits itself.
*/	
quit :: (RId Message) (PSt .l .p) -> PSt .l .p
quit you ps
	=	closeProcess (snd (syncSend you Quit ps))
